<?php
session_start();

// Verificar que el usuario esté logueado
if (!isset($_SESSION['id'])) {
    header('Location: ../login.html');
    exit();
}

// Obtener parámetros
$curso = isset($_GET['curso']) ? $_GET['curso'] : '';
$unidad = isset($_GET['unidad']) ? $_GET['unidad'] : '';

if (empty($curso) || empty($unidad)) {
    die('Error: Faltan parámetros (curso o unidad)');
}

// Normalizar unidad
$unidad = str_pad($unidad, 2, '0', STR_PAD_LEFT);

// Conectar a la base de datos
require_once '../conexion.php';

// Nombre de la tabla
$nombre_tabla = "avance_{$curso}_u{$unidad}";

// Intentar obtener las columnas de la tabla con comentarios
try {
    // Query para obtener información de las columnas con comentarios
    $sql = "SELECT 
                COLUMN_NAME as 'Field',
                COLUMN_TYPE as 'Type',
                IS_NULLABLE as 'Null',
                COLUMN_KEY as 'Key',
                COLUMN_DEFAULT as 'Default',
                EXTRA as 'Extra',
                COLUMN_COMMENT as 'Comment'
            FROM INFORMATION_SCHEMA.COLUMNS 
            WHERE TABLE_SCHEMA = DATABASE() 
            AND TABLE_NAME = :tabla 
            ORDER BY ORDINAL_POSITION";
    
    $stmt = $dsn->prepare($sql);
    $stmt->bindParam(':tabla', $nombre_tabla, PDO::PARAM_STR);
    $stmt->execute();
    $columnas = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Obtener información sobre el uso de las variables
    $sql_uso = "SELECT COUNT(*) as total_registros FROM `{$nombre_tabla}`";
    $stmt_uso = $dsn->prepare($sql_uso);
    $stmt_uso->execute();
    $info_registros = $stmt_uso->fetch(PDO::FETCH_ASSOC);
    
    // Obtener valores del usuario actual
    $usuario_id = $_SESSION['id'];
    $sql_usuario = "SELECT * FROM `{$nombre_tabla}` WHERE id = :usuario_id";
    $stmt_usuario = $dsn->prepare($sql_usuario);
    $stmt_usuario->bindParam(':usuario_id', $usuario_id, PDO::PARAM_INT);
    $stmt_usuario->execute();
    $valores_usuario = $stmt_usuario->fetch(PDO::FETCH_ASSOC);
    
} catch (PDOException $e) {
    $error = $e->getMessage();
    $columnas = [];
    $valores_usuario = [];
}
?>

<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Variables de Base de Datos - <?php echo $nombre_tabla; ?></title>
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css" rel="stylesheet"/>
    <link rel="stylesheet" href="../general.css" />
    <style>
        body {
            font-family: "Nunito", sans-serif;
            background: radial-gradient(circle, rgba(67,69,112,1) 3%, rgba(35,36,57,1) 60%);
            min-height: 100vh;
            margin: 0;
            padding: 2rem;
        }

        .container {
            max-width: 1200px;
            margin: 0 auto;
            background: rgba(255,255,255,0.1);
            border-radius: 12px;
            padding: 2rem;
            box-shadow: 0 10px 30px rgba(0,0,0,0.3);
        }

        .header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 2rem;
            padding-bottom: 1rem;
            border-bottom: 2px solid rgba(255,255,255,0.2);
        }

        .header h1 {
            color: #fff;
            font-size: 2rem;
            margin: 0;
            font-family: "Tilt Neon", sans-serif;
        }

        .btn-back {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 0.75rem 1.5rem;
            border: none;
            border-radius: 8px;
            cursor: pointer;
            font-weight: 600;
            transition: all 0.3s ease;
            display: flex;
            align-items: center;
            gap: 0.5rem;
            text-decoration: none;
        }

        .btn-back:hover {
            transform: translateY(-2px);
            box-shadow: 0 5px 15px rgba(102, 126, 234, 0.4);
        }

        .info-box {
            background: rgba(79, 195, 247, 0.1);
            border-left: 4px solid #4fc3f7;
            padding: 1rem;
            border-radius: 8px;
            margin-bottom: 2rem;
        }

        .info-box h3 {
            color: #4fc3f7;
            margin: 0 0 0.5rem 0;
            font-size: 1.2rem;
        }

        .info-box p {
            color: #fff;
            margin: 0.25rem 0;
        }

        .error-box {
            background: rgba(244, 67, 54, 0.1);
            border-left: 4px solid #f44336;
            padding: 1.5rem;
            border-radius: 8px;
            color: #fff;
        }

        .error-box h3 {
            color: #f44336;
            margin: 0 0 0.5rem 0;
        }

        table {
            width: 100%;
            border-collapse: collapse;
            background: rgba(255,255,255,0.05);
            border-radius: 8px;
            overflow: hidden;
        }

        thead {
            background: rgba(79, 195, 247, 0.2);
        }

        th, td {
            padding: 1rem;
            text-align: left;
            border-bottom: 1px solid rgba(255,255,255,0.1);
        }

        th {
            color: #4fc3f7;
            font-weight: 600;
            text-transform: uppercase;
            font-size: 0.9rem;
        }

        td {
            color: #fff;
        }

        tr:hover {
            background: rgba(255,255,255,0.05);
        }

        .badge {
            display: inline-block;
            padding: 0.25rem 0.75rem;
            border-radius: 20px;
            font-size: 0.85rem;
            font-weight: 600;
        }

        .badge-primary {
            background: rgba(79, 195, 247, 0.2);
            color: #4fc3f7;
        }

        .badge-success {
            background: rgba(76, 175, 80, 0.2);
            color: #4caf50;
        }

        .badge-info {
            background: rgba(33, 150, 243, 0.2);
            color: #2196f3;
        }

        .btn-reset {
            background: rgba(244, 67, 54, 0.2);
            color: #f44336;
            border: 1px solid #f44336;
            padding: 0.4rem 0.8rem;
            border-radius: 6px;
            cursor: pointer;
            font-size: 0.85rem;
            transition: all 0.3s ease;
            display: inline-flex;
            align-items: center;
            gap: 0.4rem;
        }

        .btn-reset:hover {
            background: rgba(244, 67, 54, 0.3);
            transform: translateY(-1px);
        }

        .btn-reset:disabled {
            opacity: 0.5;
            cursor: not-allowed;
            transform: none;
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <h1><i class="fas fa-database"></i> Variables de Base de Datos</h1>
            <a href="../panel.php" class="btn-back">
                <i class="fas fa-arrow-left"></i>
                Volver al Panel
            </a>
        </div>

        <?php if (isset($error)): ?>
            <div class="error-box">
                <h3><i class="fas fa-exclamation-triangle"></i> Error</h3>
                <p>No se pudo acceder a la tabla: <?php echo htmlspecialchars($nombre_tabla); ?></p>
                <p style="margin-top: 0.5rem; font-size: 0.9rem; color: #bbb;">Error: <?php echo htmlspecialchars($error); ?></p>
                <p style="margin-top: 1rem;">Asegúrate de que la tabla exista en la base de datos. Puedes crearla usando el botón "Crear tabla" en el panel.</p>
            </div>
        <?php else: ?>
            <div class="info-box">
                <h3><i class="fas fa-info-circle"></i> Información de la Tabla</h3>
                <p><strong>Nombre de la tabla:</strong> <?php echo htmlspecialchars($nombre_tabla); ?></p>
                <p><strong>Curso:</strong> <?php echo htmlspecialchars($curso); ?></p>
                <p><strong>Unidad:</strong> <?php echo htmlspecialchars($unidad); ?></p>
                <p><strong>Total de columnas:</strong> <?php echo count($columnas); ?></p>
                <?php if (isset($info_registros)): ?>
                    <p><strong>Total de registros:</strong> <?php echo number_format($info_registros['total_registros']); ?></p>
                <?php endif; ?>
            </div>

            <table>
                <thead>
                    <tr>
                        <th>Nombre de Columna</th>
                        <th>Tipo de Dato</th>
                        <th>Nulo</th>
                        <th>Clave</th>
                        <th>Predeterminado</th>
                        <th>Extra</th>
                        <th>Comentario</th>
                        <th>Acción</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($columnas as $columna): ?>
                        <?php 
                            // Obtener el valor del usuario para esta columna
                            $valor_usuario = null;
                            if (!empty($valores_usuario) && isset($valores_usuario[$columna['Field']])) {
                                $valor_usuario = $valores_usuario[$columna['Field']];
                            }
                            
                            // Formatear el valor para mostrar
                            $valor_mostrar = '-';
                            if ($valor_usuario !== null) {
                                if (is_null($valor_usuario)) {
                                    $valor_mostrar = '<em style="color: #888;">NULL</em>';
                                } elseif (is_string($valor_usuario) && strlen($valor_usuario) > 50) {
                                    $valor_mostrar = htmlspecialchars(substr($valor_usuario, 0, 50)) . '...';
                                } else {
                                    $valor_mostrar = htmlspecialchars($valor_usuario);
                                }
                            }
                        ?>
                        <tr>
                            <td>
                                <strong><?php echo htmlspecialchars($columna['Field']); ?></strong>
                                <?php if ($columna['Key'] === 'PRI'): ?>
                                    <span class="badge badge-success">PRIMARY</span>
                                <?php elseif ($columna['Key'] === 'MUL'): ?>
                                    <span class="badge badge-info">INDEX</span>
                                <?php endif; ?>
                            </td>
                            <td>
                                <span class="badge badge-primary">
                                    <?php echo htmlspecialchars($columna['Type']); ?>
                                </span>
                            </td>
                            <td><?php echo $columna['Null'] === 'YES' ? 'Sí' : 'No'; ?></td>
                            <td><?php echo htmlspecialchars($columna['Key']); ?></td>
                            <td><?php echo $columna['Default'] !== null ? htmlspecialchars($columna['Default']) : 'NULL'; ?></td>
                            <td><?php echo $valor_mostrar; ?></td>
                            <td style="color: #aaa; font-style: italic;">
                                <?php echo !empty($columna['Comment']) ? htmlspecialchars($columna['Comment']) : '-'; ?>
                            </td>
                            <td>
                                <?php if ($columna['Key'] !== 'PRI'): ?>
                                    <button class="btn-reset" 
                                            onclick="reiniciarVariable('<?php echo htmlspecialchars($columna['Field']); ?>')"
                                            title="Reiniciar esta variable">
                                        <i class="fas fa-redo"></i>
                                        Reiniciar
                                    </button>
                                <?php else: ?>
                                    <span style="color: #888; font-size: 0.9rem;">-</span>
                                <?php endif; ?>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        <?php endif; ?>
    </div>

    <script>
        const curso = '<?php echo htmlspecialchars($curso); ?>';
        const unidad = '<?php echo htmlspecialchars($unidad); ?>';
        const usuario_id = <?php echo $_SESSION['id']; ?>;

        async function reiniciarVariable(nombreVariable) {
            if (!confirm(`¿Estás seguro de reiniciar la variable "${nombreVariable}"?`)) {
                return;
            }

            const btn = event.target.closest('.btn-reset');
            const originalHTML = btn.innerHTML;
            btn.disabled = true;
            btn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Procesando...';

            try {
                const response = await fetch('reiniciar_variable.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json'
                    },
                    body: JSON.stringify({
                        curso: curso,
                        unidad: unidad,
                        variable: nombreVariable,
                        usuario_id: usuario_id
                    })
                });

                const data = await response.json();

                if (data.success) {
                    alert('✅ Variable reiniciada exitosamente');
                    location.reload();
                } else {
                    alert('❌ Error: ' + (data.message || 'Error desconocido'));
                    btn.disabled = false;
                    btn.innerHTML = originalHTML;
                }
            } catch (error) {
                console.error('Error:', error);
                alert('❌ Error de conexión al reiniciar la variable');
                btn.disabled = false;
                btn.innerHTML = originalHTML;
            }
        }
    </script>
</body>
</html>
