<?php
/**
 * Aumenta IDs en la tabla de inscripción de cursos desde el 101 en adelante
 * Crea nuevos registros con todos los campos de cursos en NULL
 */

// Configurar headers para JSON
header('Content-Type: application/json; charset=utf-8');
header('Cache-Control: no-cache, no-store, must-revalidate');
header('Pragma: no-cache');
header('Expires: 0');

// Verificar que sea una petición POST
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['error' => 'Método no permitido']);
    exit;
}

// Obtener el contenido JSON
$input = file_get_contents('php://input');
$data = json_decode($input, true);

if (!$data) {
    http_response_code(400);
    echo json_encode(['error' => 'Datos JSON inválidos']);
    exit;
}

// Extraer datos
$dbHost = $data['dbHost'] ?? '';
$dbName = $data['dbName'] ?? '';
$dbUser = $data['dbUser'] ?? '';
$dbPassword = $data['dbPassword'] ?? '';
$dbPort = $data['dbPort'] ?? '3306';
$nombreTabla = $data['nombreTabla'] ?? '';
$securityKey = $data['securityKey'] ?? '';
$idInicio = intval($data['idInicio'] ?? 0);
$idFinal = intval($data['idFinal'] ?? 0);

// Validar datos requeridos
if (empty($dbHost) || empty($dbName) || empty($dbUser) || empty($dbPort) || empty($nombreTabla)) {
    http_response_code(400);
    echo json_encode(['error' => 'Faltan datos requeridos']);
    exit;
}

// Validar parámetros de aumento de IDs
if ($idInicio < 101 || $idFinal < $idInicio) {
    http_response_code(400);
    echo json_encode(['error' => 'ID de inicio debe ser al menos 101 y ID final debe ser mayor']);
    exit;
}

if (($idFinal - $idInicio + 1) > 1000) {
    http_response_code(400);
    echo json_encode(['error' => 'Máximo 1000 IDs por operación']);
    exit;
}

// Validar clave de seguridad
$expectedKey = 'clave_secreta_123'; // Cambiar por una clave más segura en producción
if ($securityKey !== $expectedKey) {
    http_response_code(403);
    echo json_encode(['error' => 'Clave de seguridad incorrecta']);
    exit;
}

// Función para registrar logs
function logAumentarIds($message) {
    $logFile = '../logs/aumentar_ids_inscripcion_cursos.log';
    $timestamp = date('Y-m-d H:i:s');
    $logMessage = "[$timestamp] $message\n";
    
    // Crear directorio de logs si no existe
    if (!is_dir('../logs')) {
        mkdir('../logs', 0755, true);
    }
    
    file_put_contents($logFile, $logMessage, FILE_APPEND | LOCK_EX);
}

try {
    // Registrar intento de aumento de IDs
    logAumentarIds("Aumento de IDs - Host: $dbHost, DB: $dbName, Tabla: $nombreTabla, Rango: $idInicio-$idFinal");
    
    // Crear conexión PDO
    $dsn = "mysql:host=$dbHost;port=$dbPort;dbname=$dbName;charset=utf8mb4";
    $options = [
        PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
        PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
        PDO::ATTR_EMULATE_PREPARES => false,
    ];
    
    $pdo = new PDO($dsn, $dbUser, $dbPassword, $options);
    
    // Verificar si la tabla existe
    $stmt = $pdo->prepare("SELECT COUNT(*) as existe FROM information_schema.tables WHERE table_schema = ? AND table_name = ?");
    $stmt->execute([$dbName, $nombreTabla]);
    $result = $stmt->fetch();
    
    if ($result['existe'] == 0) {
        logAumentarIds("Error: Tabla no existe - $nombreTabla");
        
        echo json_encode([
            'success' => false,
            'message' => 'La tabla no existe en la base de datos',
            'timestamp' => date('Y-m-d H:i:s')
        ]);
        exit;
    }
    
    // Verificar si ya existen algunos de los IDs que se quieren insertar
    $stmt = $pdo->prepare("SELECT COUNT(*) as existentes FROM `$nombreTabla` WHERE id BETWEEN ? AND ?");
    $stmt->execute([$idInicio, $idFinal]);
    $existentes = $stmt->fetch()['existentes'];
    
    if ($existentes > 0) {
        logAumentarIds("Advertencia: Ya existen $existentes IDs en el rango $idInicio-$idFinal");
        
        echo json_encode([
            'success' => false,
            'message' => "Ya existen $existentes IDs en el rango $idInicio-$idFinal. Por favor, elige otro rango de IDs.",
            'timestamp' => date('Y-m-d H:i:s')
        ]);
        exit;
    }
    
    // Comenzar transacción
    $pdo->beginTransaction();
    
    // Preparar la consulta de inserción
    $sql = "INSERT INTO `$nombreTabla` (id, c001, c002, c003, c004, c005, c006, c007, c008, c009, c010) VALUES (?, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL)";
    $stmt = $pdo->prepare($sql);
    
    $registrosInsertados = 0;
    $idsInsertados = [];
    
    // Insertar registros uno por uno
    for ($i = $idInicio; $i <= $idFinal; $i++) {
        $stmt->execute([$i]);
        $registrosInsertados++;
        $idsInsertados[] = $i;
    }
    
    // Confirmar transacción
    $pdo->commit();
    
    logAumentarIds("Aumento de IDs exitoso - Registros insertados: $registrosInsertados, IDs: " . implode(', ', $idsInsertados));
    
    // Retornar resultado exitoso
    echo json_encode([
        'success' => true,
        'message' => "IDs aumentados exitosamente de $idInicio hasta $idFinal",
        'registrosInsertados' => $registrosInsertados,
        'idInicio' => $idInicio,
        'idFinal' => $idFinal,
        'idsInsertados' => $idsInsertados,
        'timestamp' => date('Y-m-d H:i:s')
    ]);
    
    // Cerrar conexión
    $pdo = null;
    
} catch (PDOException $e) {
    // Error de base de datos
    if (isset($pdo) && $pdo->inTransaction()) {
        $pdo->rollBack();
    }
    
    logAumentarIds("Error de base de datos: " . $e->getMessage());
    
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'error' => 'Error de base de datos',
        'message' => $e->getMessage(),
        'timestamp' => date('Y-m-d H:i:s')
    ]);
    
} catch (Exception $e) {
    // Error general
    if (isset($pdo) && $pdo->inTransaction()) {
        $pdo->rollBack();
    }
    
    logAumentarIds("Error general: " . $e->getMessage());
    
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'error' => 'Error interno del servidor',
        'message' => $e->getMessage(),
        'timestamp' => date('Y-m-d H:i:s')
    ]);
}
?> 