<?php
session_start();

/**
 * Verifica si la tabla inscripcion_cursos existe en la base de datos
 * @return bool True si la tabla existe, False si no existe
 */
function verificarExistenciaTabla() {
    try {
        // Cargar configuración de base de datos
        if (file_exists('../config.php')) {
            require_once '../config.php';
            
            // Obtener configuración del ambiente de producción
            if (isset($config) && isset($config['production'])) {
                $db_config = $config['production'];
                
                // Crear conexión PDO
                $dsn = "mysql:host=" . $db_config['db_host'] . ";dbname=" . $db_config['db_name'] . ";charset=utf8mb4";
                $pdo = new PDO($dsn, $db_config['db_user'], $db_config['db_pass']);
                $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
                
                // Verificar si la tabla existe
                $stmt = $pdo->prepare("SELECT COUNT(*) as existe FROM information_schema.tables WHERE table_schema = ? AND table_name = ?");
                $stmt->execute([$db_config['db_name'], 'inscripcion_cursos']);
                $result = $stmt->fetch();
                
                return $result['existe'] > 0;
            }
        }
        
        // Si no se puede conectar o no hay configuración, retornar false
        return false;
        
    } catch (Exception $e) {
        // En caso de error, retornar false
        return false;
    }
}
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8" />
    <meta name="viewport" content="width=device-width, initial-scale=1.0" />
    <title>Checklist de Archivos - Ruta Formativa CCBB</title>
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css" rel="stylesheet"/>
    <link rel="stylesheet" href="ayuda.css" />
    <link rel="stylesheet" href="../general.css" />
    
    <style>
        /* Estilos para el switch toggle */
        .switch {
            position: relative;
            display: inline-block;
            width: 50px;
            height: 24px;
        }
        
        .switch input {
            opacity: 0;
            width: 0;
            height: 0;
        }
        
        .slider {
            position: absolute;
            cursor: pointer;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background-color: #ccc;
            transition: .4s;
            border-radius: 24px;
        }
        
        .slider:before {
            position: absolute;
            content: "";
            height: 18px;
            width: 18px;
            left: 3px;
            bottom: 3px;
            background-color: white;
            transition: .4s;
            border-radius: 50%;
        }
        
        input:checked + .slider {
            background-color: #29b6f6;
        }
        
        input:checked + .slider:before {
            transform: translateX(26px);
        }
        
        .manual-toggle {
            vertical-align: middle;
        }
        
        .manual-toggle .switch {
            margin: 0;
        }
    </style>
</head>

<body>
    <!-- Botón de regreso -->
    <div class="back-button">
        <a href="../menu.php" class="boton-azul">
            <i class="fas fa-arrow-left"></i>
            <span>Volver al Menú</span>
        </a>
    </div>

    <!-- Contenido principal -->
    <div class="checklist-container">
        <div class="checklist-header">
            <h1 class="checklist-title">
                <i class="fas fa-tasks"></i>
                Checklist de Archivos
            </h1>
            <p class="checklist-subtitle">
                Seguimiento del desarrollo de la plataforma Ruta Formativa CCBB
            </p>
            <p class="last-update">
                Última actualización: <?php echo date('d/m/Y H:i'); ?>
            </p>
        </div>

        <!-- Fase 1: Página Principal -->
        <div class="phase-section">
            <h2 class="phase-title">
                <i class="fas fa-home"></i> Fase 1: Página Principal
            </h2>
            <p class="phase-description">
                Archivos base para la página de inicio de la plataforma
            </p>
            
            <?php
            $fase1 = [
                ['nombre' => 'index.html', 'descripcion' => 'Página principal de la plataforma', 'funcion' => 'generarIndexHTML'],
                ['nombre' => 'general.css', 'descripcion' => 'Estilos generales y botones', 'funcion' => 'generarGeneralCSS'],
                ['nombre' => 'index.css', 'descripcion' => 'Estilos específicos de la página principal', 'funcion' => 'generarIndexCSS']
            ];
            
            foreach ($fase1 as $archivo) {
                $ruta = '../' . $archivo['nombre'];
                $existe = file_exists($ruta);
                $statusClass = $existe ? 'exists' : 'missing';
                $statusText = $existe ? 'EXISTE' : 'NO EXISTE';
                
                echo '<div class="file-item ' . ($existe ? 'exists' : 'missing') . '">';
                echo '<div class="file-info">';
                echo '<div class="file-name">' . $archivo['nombre'] . '</div>';
                echo '<div class="file-description">' . $archivo['descripcion'] . '</div>';
                echo '</div>';
                echo '<div class="file-status">';
                echo '<span class="status-icon ' . $statusClass . '">' . $statusText . '</span>';
                echo '<button class="generator-link" onclick="' . $archivo['funcion'] . '()">';
                echo '<i class="fas fa-magic"></i> Generar';
                echo '</button>';
                if ($archivo['nombre'] === 'index.html') {
                    echo '<a href="../index.html" target="_blank" class="web-link">';
                    echo '<i class="fas fa-external-link-alt"></i> Ver Web';
                    echo '</a>';
                }
                echo '</div>';
                echo '</div>';
            }
            ?>
        </div>

        <!-- Fase 2: Sistema de Login -->
        <div class="phase-section">
            <h2 class="phase-title">
                <i class="fas fa-sign-in-alt"></i> Fase 2: Sistema de Login
            </h2>
            <p class="phase-description">
                Sistema completo de autenticación y gestión de usuarios
            </p>
            
            <?php
            $fase2 = [
                ['nombre' => 'login.html', 'descripcion' => 'Página de inicio de sesión', 'funcion' => 'generarLoginHTML'],
                ['nombre' => 'login.css', 'descripcion' => 'Estilos del formulario de login', 'funcion' => 'generarLoginCSS'],
                ['nombre' => 'config.php', 'descripcion' => 'Configuración de base de datos', 'funcion' => 'generarConfigPHP'],
                ['nombre' => 'conexion.php', 'descripcion' => 'Conexión segura a la base de datos', 'funcion' => 'generarConexionPHP'],
                ['nombre' => 'login.php', 'descripcion' => 'Procesamiento del formulario de login', 'funcion' => 'generarLoginPHP'],
                ['nombre' => 'session_start.php', 'descripcion' => 'Gestión de sesiones de usuario', 'funcion' => 'generarSessionStartPHP']
            ];
            
            foreach ($fase2 as $archivo) {
                $ruta = '../' . $archivo['nombre'];
                $existe = file_exists($ruta);
                $statusClass = $existe ? 'exists' : 'missing';
                $statusText = $existe ? 'EXISTE' : 'NO EXISTE';
                
                echo '<div class="file-item ' . ($existe ? 'exists' : 'missing') . '">';
                echo '<div class="file-info">';
                echo '<div class="file-name">' . $archivo['nombre'] . '</div>';
                echo '<div class="file-description">' . $archivo['descripcion'] . '</div>';
                echo '</div>';
                echo '<div class="file-status">';
                echo '<span class="status-icon ' . $statusClass . '">' . $statusText . '</span>';
                echo '<button class="generator-link" onclick="' . $archivo['funcion'] . '()">';
                echo '<i class="fas fa-magic"></i> Generar';
                echo '</button>';
                echo '</div>';
                echo '</div>';
            }
            ?>
        </div>

        <!-- Fase 3: Panel Principal -->
        <div class="phase-section">
            <h2 class="phase-title">
                <i class="fas fa-th-large"></i> Fase 3: Panel Principal
            </h2>
            <p class="phase-description">
                Dashboard principal y navegación de la plataforma
            </p>
            
            <?php
            $fase3 = [
                ['nombre' => 'configuracion_cursos.php', 'descripcion' => 'Configuración de información de cursos', 'funcion' => 'generarConfiguracionCursosPHP'],
                ['nombre' => 'lectura_inscripcion_cursos.php', 'descripcion' => 'Lectura de configuración de inscripción de cursos del usuario', 'funcion' => 'generarLecturaCursosPHP'],
                ['nombre' => 'crear_tabla_inscripcion_cursos.sql', 'descripcion' => 'Script SQL para crear tabla de inscripción de cursos', 'funcion' => 'generarTablaInscripcionCursosSQL', 'es_tabla' => true],
                ['nombre' => 'editar_tabla_inscripcion_cursos.sql', 'descripcion' => 'Script SQL para editar tabla de inscripción de cursos', 'funcion' => 'generarEditarTablaInscripcionCursosSQL', 'es_manual' => true],
                ['nombre' => 'leer_rol.php', 'descripcion' => 'Generador para crear leer_rol.php', 'funcion' => 'generarCrearLeerRolPHP'],
                ['nombre' => 'header.css', 'descripcion' => 'Estilos del encabezado', 'funcion' => 'generarHeaderCSS'],
                ['nombre' => 'menu.css', 'descripcion' => 'Estilos del panel principal', 'funcion' => 'generarMenuCSS'],
                ['nombre' => 'menu.php', 'descripcion' => 'Panel principal con lista de cursos', 'funcion' => 'generarMenuPHP'],
                ['nombre' => 'logout.php', 'descripcion' => 'Cerrar sesión y redirigir al login', 'funcion' => 'generarLogoutPHP'],
            ];
            
            foreach ($fase3 as $archivo) {
                if (isset($archivo['es_tabla']) && $archivo['es_tabla']) {
                    // Es una tabla de base de datos, verificar su existencia
                    $existeTabla = verificarExistenciaTabla();
                    $statusClass = $existeTabla ? 'exists' : 'missing';
                    $statusText = $existeTabla ? 'EXISTE' : 'NO EXISTE';
                    $descripcion = $existeTabla ? 'Tabla inscripcion_cursos existe en la base de datos' : 'Tabla inscripcion_cursos no existe en la base de datos';
                } elseif (isset($archivo['es_manual']) && $archivo['es_manual']) {
                    // Es un recurso con control manual del estado - por defecto NO EXISTE
                    $existeTabla = false; // Por defecto no existe
                    $statusClass = 'missing';
                    $statusText = 'NO EXISTE';
                    $descripcion = $archivo['descripcion'];
                } else {
                    // Es un archivo normal, verificar su existencia
                    $ruta = '../' . $archivo['nombre'];
                    $existeTabla = file_exists($ruta);
                    $statusClass = $existeTabla ? 'exists' : 'missing';
                    $statusText = $existeTabla ? 'EXISTE' : 'NO EXISTE';
                    $descripcion = $archivo['descripcion'];
                }
                
                echo '<div class="file-item ' . ($existeTabla ? 'exists' : 'missing') . '" data-filename="' . $archivo['nombre'] . '">';
                echo '<div class="file-info">';
                echo '<div class="file-name">' . $archivo['nombre'] . '</div>';
                echo '<div class="file-description">' . $descripcion . '</div>';
                echo '</div>';
                echo '<div class="file-status">';
                echo '<span class="status-icon ' . $statusClass . '">' . $statusText . '</span>';
                
                // Si es manual, agregar el switch
                if (isset($archivo['es_manual']) && $archivo['es_manual']) {
                    echo '<div class="manual-toggle" style="margin: 0 10px; display: inline-block;">';
                    echo '<label class="switch">';
                    echo '<input type="checkbox" id="toggle_' . str_replace('.', '_', $archivo['nombre']) . '" ' . ($existeTabla ? 'checked' : '') . ' onchange="toggleManualStatus(\'' . $archivo['nombre'] . '\', this.checked)">';
                    echo '<span class="slider round"></span>';
                    echo '</label>';
                    echo '</div>';
                }
                
                echo '<button class="generator-link" onclick="' . $archivo['funcion'] . '()">';
                echo '<i class="fas fa-magic"></i> Generar';
                echo '</button>';
                echo '</div>';
                echo '</div>';
            }
            ?>
        </div>

        <!-- Fase 4: Panel de Control -->
        <div class="phase-section">
            <h2 class="phase-title">
                <i class="fas fa-cogs"></i> Fase 4: Panel de Control
            </h2>
            <p class="phase-description">
                Acceso al panel de gestión de cursos, unidades y actividades
            </p>
            
            <div class="file-item exists">
                <div class="file-info">
                    <div class="file-name">panel.php</div>
                    <div class="file-description">Panel de control para gestión de cursos, unidades y actividades</div>
                </div>
                <div class="file-status">
                    <a href="../panel.php" class="web-link" id="acceder-panel">
                        <i class="fas fa-cogs"></i> Acceder al Panel
                    </a>
                </div>
            </div>
        </div>
    </div>

    <script>
        // Función para actualizar el estado de archivos manuales al cargar la página
        function actualizarEstadosManuales() {
            const archivosManuales = [
                'editar_tabla_inscripcion_cursos.sql'
            ];
            
            archivosManuales.forEach(filename => {
                const storageKey = 'manual_' + filename.replace(/\./g, '_');
                const estado = localStorage.getItem(storageKey);
                
                if (estado === 'exists') {
                    // Buscar el elemento y actualizarlo
                    const fileItem = document.querySelector(`[data-filename="${filename}"]`);
                    if (fileItem) {
                        fileItem.classList.remove('missing');
                        fileItem.classList.add('exists');
                        
                        // Actualizar el status
                        const statusIcon = fileItem.querySelector('.status-icon');
                        if (statusIcon) {
                            statusIcon.textContent = 'EXISTE';
                            statusIcon.className = 'status-icon exists';
                        }
                        
                        // Actualizar el switch
                        const toggle = fileItem.querySelector('input[type="checkbox"]');
                        if (toggle) {
                            toggle.checked = true;
                        }
                    }
                }
            });
        }
        
        // Ejecutar cuando se carga la página
        document.addEventListener('DOMContentLoaded', actualizarEstadosManuales);
        
        // Función para generar index.html - redirige al generador
        function generarIndexHTML() {
            window.location.href = 'crear_index.html';
        }
        
        // Función para generar general.css
        function generarGeneralCSS() {
            // Contenido fijo del archivo general.css
            const cssContent = `/* general.css */
@import url("https://fonts.googleapis.com/css2?family=Nunito:wght@300;400;500;600;700;800&family=Tilt+Neon&display=swap");

:root {
    /* Defino la variable con el degradado completo */
    --gradiente-blue: linear-gradient(to bottom,#0288d1 0%,#29b6f6 50%,#4fc3f7 100%);
    --gradiente-green: linear-gradient(to bottom, #76a73f 0%,#338f76 100%);
    --gradiente-orange: linear-gradient(to bottom, #ffb74d 0%, #ff9800 50%, #f57c00 100%);
    --gradiente-gray: linear-gradient(to bottom,#f5f5f5 0%, #cccccc 50%, #999999 100%);
    --gradiente-rojo: linear-gradient(90deg, #e52d27 0%, #b31217 100%);
    --gradiente-multicolor: linear-gradient(90deg,#76a73f 0%,#338f76 40%,#378da7 50%,#7598f2 70%,#9076ec 100%);
    /* Variable para color rojo sólido */
    --color-rojo: #e52d27;
}

.content {
    flex: 1;
    display: grid;
    grid-template-rows: auto auto 1fr auto;
    row-gap: 0.15rem;
    align-items: start;
    justify-items: center;
    padding: 1rem;
    text-align: center;
}

.boton-verde {
    display: inline-flex;
    align-items: center;
    justify-content: center;
    color: #fff;
    background: var(--gradiente-green);
    padding: 0.5rem 1rem;
    border-radius: 8px;
    text-decoration: none;
    font-weight: 500;
    transition: background-position 0.3s ease, transform 0.3s ease;
    margin-top: 0rem;
    font-size: 1rem;
    position: relative;
    border: none;
    outline: none;
    box-sizing: border-box;
    line-height: 1;
    height: 2rem;
    -webkit-appearance: none;
    -moz-appearance: none;
    appearance: none;
}

.boton-verde:hover {
    transform: scale(1.05);
}

.boton-gris {
    display: inline-block;
    color: #fff;
    background: var(--gradiente-gray);
    padding: 0.5rem 1rem;
    border-radius: 8px;
    text-decoration: none;
    font-weight: 500;
    transition: background-position 0.3s ease, transform 0.3s ease;
    margin-top: 0.5rem;
    font-size: 1rem;
}

.boton-gris:hover {
    transform: scale(1.05);
}

.boton-rojo {
    display: inline-block;
    color: #fff;
    background: var(--gradiente-rojo);
    padding: 0.5rem 1rem;
    border-radius: 8px;
    text-decoration: none;
    font-weight: 500;
    transition: background-position 0.3s ease, transform 0.3s ease;
    margin-top: 0.5rem;
    font-size: 1rem;
}

.boton-rojo:hover {
    transform: scale(1.05);
}

.boton-anaranjado {
    display: inline-flex;
    color: #fff;
    background: var(--gradiente-orange);
    padding: 0.5rem 1rem;
    border-radius: 8px;
    text-decoration: none;
    font-weight: 500;
    transition: background-position 0.3s ease, transform 0.3s ease;
    margin-top: 0.5rem;
    font-size: 1rem;
    position: relative;
    align-items: center;
}

.boton-anaranjado:hover {
    transform: scale(1.05);
}

.boton-celeste {
    display: inline-flex;
    align-items: center;
    justify-content: center;
    color: #fff;
    background: #29b6f6;
    padding: 0.5rem 1rem;
    border-radius: 8px;
    text-decoration: none;
    font-weight: 500;
    transition: background-position 0.3s ease, transform 0.3s ease;
    margin-top: 0rem;
    font-size: 1rem;
    position: relative;
    border: none;
    outline: none;
    box-sizing: border-box;
    line-height: 1;
    height: 2rem;
    -webkit-appearance: none;
    -moz-appearance: none;
    appearance: none;
}

.boton-celeste:hover {
    transform: scale(1.05);
}

.boton-naranja {
    display: inline-flex;
    align-items: center;
    justify-content: center;
    color: #fff;
    background: #ff9800;
    padding: 0.5rem 1rem;
    border-radius: 8px;
    text-decoration: none;
    font-weight: 500;
    transition: background-position 0.3s ease, transform 0.3s ease;
    margin-top: 0rem;
    font-size: 1rem;
    position: relative;
    border: none;
    outline: none;
    box-sizing: border-box;
    line-height: 1;
    height: 2rem;
    -webkit-appearance: none;
    -moz-appearance: none;
    appearance: none;
}

.boton-naranja:hover {
    transform: scale(1.05);
}

.boton-azul {
    display: inline-flex;
    align-items: center;
    justify-content: center;
    color: #fff;
    background: var(--gradiente-blue);
    padding: 0.5rem 1rem;
    border-radius: 8px;
    text-decoration: none;
    font-weight: 500;
    transition: background-position 0.3s ease, transform 0.3s ease;
    margin-top: 0rem;
    font-size: 1rem;
    position: relative;
    border: none;
    outline: none;
    box-sizing: border-box;
    line-height: 1;
    height: 2rem;
    -webkit-appearance: none;
    -moz-appearance: none;
    appearance: none;
}

.boton-azul:hover {
    transform: scale(1.05);
}

.boton-no-disponible {
    display: inline-flex;
    align-items: center;
    justify-content: center;
    color: #8F8F96;
    background: #5F5F76;
    padding: 0.5rem 1rem;
    border-radius: 8px;
    text-decoration: none;
    font-weight: 500;
    transition: background-position 0.3s ease, transform 0.3s ease;
    margin-top: 0rem;
    font-size: 1rem;
    position: relative;
    border: none;
    outline: none;
    box-sizing: border-box;
    line-height: 1;
    height: 2rem;
    -webkit-appearance: none;
    -moz-appearance: none;
    appearance: none;
    cursor: not-allowed;
    opacity: 0.7;
}

.boton-blanco {
    display: inline-flex;
    align-items: center;
    justify-content: center;
    color: #29b6f6;
    background: #fff;
    padding: 0.5rem 1rem;
    border-radius: 8px;
    text-decoration: none;
    font-weight: 500;
    transition: background-position 0.3s ease, transform 0.3s ease;
    margin-top: 0rem;
    font-size: 1rem;
    position: relative;
    border: 2px solid #29b6f6;
    outline: none;
    box-sizing: border-box;
    line-height: 1;
    height: 2rem;
    -webkit-appearance: none;
    -moz-appearance: none;
    appearance: none;
}

.boton-blanco:hover {
    transform: scale(1.05);
}`;

            // Enviar el contenido al servidor para guardarlo
            fetch('guardar_general_css.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/x-www-form-urlencoded',
                },
                body: 'cssContent=' + encodeURIComponent(cssContent)
            })
            .then(response => response.text())
            .then(result => {
                if (result.includes('success')) {
                    alert('Archivo general.css generado y guardado exitosamente');
                    // Recargar la página para actualizar el estado
                    location.reload();
                } else {
                    alert('Error al generar el archivo: ' + result);
                }
            })
            .catch(error => {
                console.error('Error:', error);
                alert('Error al generar el archivo: ' + error.message);
            });
        }
        
        // Función para generar index.css
        function generarIndexCSS() {
            // Contenido fijo del archivo index.css
            const cssContent = `@import url("https://fonts.googleapis.com/css2?family=Nunito:wght@300;400;500;600;700;800&family=Tilt+Neon&display=swap");

/* ------------------------
   1. Reset y Base
------------------------- */
*, *::before, *::after {
  margin: 0;
  padding: 0;
  box-sizing: border-box;
}

html, body {
  height: 100%;
  font-family: "Nunito", sans-serif;
  scroll-behavior: smooth;
  background: radial-gradient(circle, rgba(67,69,112,1) 3%, rgba(35,36,57,1) 60%);
  /* Base font-size permanece en 16px, pero luego se ajusta en pantallas 4K */
}

img {
  width: 100%;
  display: block;
  pointer-events: none;
  user-select: none;
}

/* ------------------------
   2. Sección HOME
------------------------- */
.home {
  display: grid;
  grid-template-columns: 45% 50%;
  place-items: center;
  gap: 50px;
  min-height: 100vh;
  padding: 130px 80px;
}

.description {
  color: #fff;
  padding: 0 50px;
}

.title {
  font-family: "Tilt Neon", sans-serif;
  font-size: clamp(3rem, 5vw, 4rem);
  line-height: 1.1;
  margin-bottom: 30px;
}

.gradient-text {
  background: var(--gradiente-multicolor);
  -webkit-background-clip: text;
  -webkit-text-fill-color: transparent;
  background-clip: text;
}

.paragraph {
  font-size: clamp(0.9rem, 3vw, 1.1rem);
  line-height: 1.5;
  margin-bottom: 30px;
}

#form {
  display: flex;
  justify-content: center;
}

/* ------------------------
   3. Usuarios (Grid)
------------------------- */
.users-color-container {
  display: grid;
  grid-template-columns: repeat(4, 1fr);
  grid-template-rows: repeat(3, 1fr);
  gap: 20px;
}

.item {
  max-width: 200px;
  aspect-ratio: 1/1;
  background: #fff;
  box-shadow: 0 8px 8px rgba(0, 0, 0, 0.5), inset 0 2px 2px rgba(255, 255, 255, 0.2);
  opacity: 0;
  animation: fadeIn 0.5s forwards;
  animation-delay: calc(0.2s * var(--i));
}

/* Formas y colores específicos */
.item:nth-child(1)  { background: #67d7e1; border-radius: 50% 50% 0 50%; }
.item:nth-child(2)  { border-radius: 50% 50% 0 0; }
.item:nth-child(3)  { background: #6cc164; border-radius: 50%; }
.item:nth-child(4)  { border-radius: 0 0 0 50%; }
.item:nth-child(5)  { border-radius: 0 50% 50% 0; }
.item:nth-child(6)  { background: #8071a8; border-radius: 0 50% 50% 50%; }
.item:nth-child(7)  { border-radius: 50% 50% 0 50%; }
.item:nth-child(8)  { background: #fe7519; border-radius: 50% 0 0 50%; }
.item:nth-child(9)  { background: #f5bec3; border-radius: 0 50% 50% 0; }
.item:nth-child(10) { border-radius: 50%; }
.item:nth-child(11) { background: #fcd659; border-radius: 50% 0 50% 50%; }
.item:nth-child(12) { border-radius: 50% 0 0 0; }

@keyframes fadeIn {
  from { opacity: 0; transform: scale(0.8); }
  to   { opacity: 1; transform: scale(1); }
}

/* Para que el <a class="btn"> se comporte como tu botón */
a.boton-verde {
  display: inline-flex;           /* igual que tu botón */
  align-items: center;
  gap: 8px;
  text-decoration: none;          /* quita subrayado */
}

/* Mantiene el hover degradado */
a.boton-verde:hover {
  background-position: right;
}

.mensaje-error {
    color: var(--color-rojo);
    font-size: 0.9rem;
    margin-bottom: 1rem;
    min-height: 20px;
}

/* ------------------------
   4. Responsive
------------------------- */
@media (max-width: 1024px) {
  .home {
    grid-template-columns: 1fr;
    grid-template-rows: auto auto;
    padding: 100px 40px;
  }
  .description {
    text-align: center;
    padding: 0;
  }
}

@media (max-width: 480px) {
  .home {
    padding: 40px 20px;
    gap: 30px;
  }
  .boton-verde {
    width: 100%;
    justify-content: center;
  }
}

/* ------------------------
   5. Ajustes para 4K
------------------------- */
@media (min-width: 2560px) {
  html { font-size: 18px; }
  .title {
    font-size: clamp(4rem, 4vw, 6rem);
  }
  .paragraph {
    font-size: clamp(1rem, 2vw, 1.5rem);
  }
  .boton-verde {
    font-size: 1.25rem;
    padding: 16px 24px;
  }
}`;

            // Enviar el contenido al servidor para guardarlo
            fetch('guardar_index_css.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/x-www-form-urlencoded',
                },
                body: 'cssContent=' + encodeURIComponent(cssContent)
            })
            .then(response => response.text())
            .then(result => {
                if (result.includes('success')) {
                    alert('Archivo index.css generado y guardado exitosamente');
                    // Recargar la página para actualizar el estado
                    location.reload();
                } else {
                    alert('Error al generar el archivo: ' + result);
                }
            })
            .catch(error => {
                console.error('Error:', error);
                alert('Error al generar el archivo: ' + error.message);
            });
        }
        
        // Función para generar login.html
        function generarLoginHTML() {
            window.location.href = 'crear_login.html';
        }
        
        // Función para generar login.css
        function generarLoginCSS() {
            // Contenido fijo del archivo login.css
            const cssContent = `/* Configuración base para el login */
* {
  margin: 0;
  padding: 0;
  box-sizing: border-box;
}

body {
  font-family: "Nunito", sans-serif;
  overflow-x: hidden; /* Evitar scroll horizontal por las animaciones */
}

/* Contenedor de personajes animados */
.personajes-container {
  position: fixed;
  top: 0;
  left: 0;
  width: 100%;
  height: 100%;
  pointer-events: none;
  z-index: 1000;
}

/* Estilos base para personajes */
.personaje {
  position: absolute;
  pointer-events: none;
  z-index: 1001;
  opacity: 0;
  transform: scale(0.8);
  transition: all 0.5s ease;
}

.personaje img {
  width: 80px;
  height: 80px;
  border-radius: 50%;
  box-shadow: 0 4px 15px rgba(118, 167, 63, 0.3);
  border: 3px solid rgba(118, 167, 63, 0.5);
  animation: float 3s ease-in-out infinite;
}

/* Animación de flotación */
@keyframes float {
  0%, 100% { transform: translateY(0px) rotate(0deg); }
  50% { transform: translateY(-10px) rotate(5deg); }
}

/* Globos de diálogo */
.globo-dialogo {
  position: absolute;
  background: linear-gradient(135deg, #76a73f, #9076ec);
  color: white;
  padding: 8px 12px;
  border-radius: 15px;
  font-size: 12px;
  font-weight: 600;
  white-space: nowrap;
  box-shadow: 0 4px 12px rgba(0,0,0,0.3);
  opacity: 0;
  transform: scale(0.5);
  transition: all 0.3s ease;
}

.globo-dialogo::before {
  content: '';
  position: absolute;
  width: 0;
  height: 0;
  border-style: solid;
}

/* Personaje 1 - Esquina superior izquierda */
.personaje-1 {
  top: 20px;
  left: 20px;
  animation: aparecer1 1s ease 0.5s forwards, moverIzquierda 8s ease-in-out 2s infinite;
}

.globo-1 {
  bottom: -45px;
  left: -10px;
  animation: mostrarGlobo 0.5s ease 2s forwards, ocultarGlobo 0.5s ease 4s forwards;
}

.globo-1::before {
  top: -8px;
  left: 20px;
  border-left: 8px solid transparent;
  border-right: 8px solid transparent;
  border-bottom: 8px solid #76a73f;
}

/* Personaje 2 - Esquina superior derecha */
.personaje-2 {
  top: 20px;
  right: 20px;
  animation: aparecer2 1s ease 1.5s forwards, moverDerecha 10s ease-in-out 3s infinite;
}

.globo-2 {
  bottom: -45px;
  right: -10px;
  animation: mostrarGlobo 0.5s ease 4.5s forwards, ocultarGlobo 0.5s ease 7s forwards;
}

.globo-2::before {
  top: -8px;
  right: 20px;
  border-left: 8px solid transparent;
  border-right: 8px solid transparent;
  border-bottom: 8px solid #76a73f;
}

/* Personaje 3 - Flotante central */
.personaje-3 {
  top: 50%;
  left: 50%;
  transform: translate(-50%, -50%) scale(0.8);
  animation: aparecer3 1s ease 2.5s forwards, moverCentral 12s ease-in-out 4s infinite;
}

.globo-3 {
  top: -45px;
  left: 50%;
  transform: translateX(-50%);
  animation: mostrarGlobo 0.5s ease 7.5s forwards, ocultarGlobo 0.5s ease 10s forwards;
}

.globo-3::before {
  bottom: -8px;
  left: 50%;
  transform: translateX(-50%);
  border-left: 8px solid transparent;
  border-right: 8px solid transparent;
  border-top: 8px solid #76a73f;
}

/* Animaciones de aparición */
@keyframes aparecer1 {
  from {
    opacity: 0;
    transform: scale(0.3) translateX(-100px);
  }
  to {
    opacity: 1;
    transform: scale(1) translateX(0);
  }
}

@keyframes aparecer2 {
  from {
    opacity: 0;
    transform: scale(0.3) translateX(100px);
  }
  to {
    opacity: 1;
    transform: scale(1) translateX(0);
  }
}

@keyframes aparecer3 {
  from {
    opacity: 0;
    transform: translate(-50%, -50%) scale(0.3) translateY(100px);
  }
  to {
    opacity: 1;
    transform: translate(-50%, -50%) scale(1.2) translateY(0);
  }
}

/* Animaciones de movimiento */
@keyframes moverIzquierda {
  0%, 100% { transform: translateX(0) translateY(0) rotate(0deg); }
  25% { transform: translateX(30px) translateY(-20px) rotate(10deg); }
  50% { transform: translateX(0) translateY(-40px) rotate(0deg); }
  75% { transform: translateX(-20px) translateY(-20px) rotate(-5deg); }
}

@keyframes moverDerecha {
  0%, 100% { transform: translateX(0) translateY(0) rotate(0deg); }
  20% { transform: translateX(-40px) translateY(-15px) rotate(-8deg); }
  40% { transform: translateX(-20px) translateY(-35px) rotate(5deg); }
  60% { transform: translateX(-50px) translateY(-10px) rotate(-10deg); }
  80% { transform: translateX(-10px) translateY(-25px) rotate(3deg); }
}

@keyframes moverCentral {
  0%, 100% { transform: translate(-50%, -50%) scale(1.2) rotate(0deg); }
  16% { transform: translate(-60%, -40%) scale(1.1) rotate(15deg); }
  33% { transform: translate(-40%, -60%) scale(1.3) rotate(-10deg); }
  50% { transform: translate(-50%, -30%) scale(1.0) rotate(20deg); }
  66% { transform: translate(-30%, -50%) scale(1.2) rotate(-15deg); }
  83% { transform: translate(-55%, -45%) scale(1.1) rotate(10deg); }
}

/* Animaciones de globos */
@keyframes mostrarGlobo {
  from {
    opacity: 0;
    transform: scale(0.5) translateY(10px);
  }
  to {
    opacity: 1;
    transform: scale(1) translateY(0);
  }
}

@keyframes ocultarGlobo {
  from {
    opacity: 1;
    transform: scale(1) translateY(0);
  }
  to {
    opacity: 0;
    transform: scale(0.5) translateY(-10px);
  }
}

/* Efectos adicionales para interacción */
@keyframes pulso {
  0%, 100% { transform: scale(1); }
  50% { transform: scale(1.1); }
}

/* Responsive para móviles */
@media (max-width: 768px) {
  .personaje img {
    width: 60px;
    height: 60px;
  }
  
  .globo-dialogo {
    font-size: 10px;
    padding: 6px 10px;
  }
  
  .personaje-1 {
    top: 10px;
    left: 10px;
  }
  
  .personaje-2 {
    top: 10px;
    right: 10px;
  }
  
  .personaje-3 {
    display: none; /* Ocultar en móviles para no saturar */
  }
}

/* Evitar que los personajes interfieran con el formulario */
.login-section {
  position: relative;
  z-index: 100;
}

/* Estilos específicos para el login */
.login-section {
  min-height: 100vh;
  display: flex;
  align-items: center;
  justify-content: center;
  background: radial-gradient(circle, rgba(67,69,112,1) 3%, rgba(35,36,57,1) 60%);
  padding: 2rem;
}
.login-container {
  background: #1f1f38;
  padding: 2rem;
  border-radius: 8px;
  box-shadow: 0 8px 16px rgba(0,0,0,0.6);
  width: 100%;
  max-width: 400px;
}
.login-container h2 {
  color: #fff;
  text-align: center;
  margin-bottom: 1.5rem;
  font-family: "Tilt Neon", sans-serif;
}
.form-group {
  margin-bottom: 1rem;
}
.form-group label {
  display: block;
  color: #ddd;
  margin-bottom: 0.5rem;
  font-family: "Nunito", sans-serif;
}
.form-group input {
  width: 100%;
  padding: 0.5rem;
  border-radius: 4px;
  border: 1px solid #555;
  background: #2a2a45;
  color: #fff;
  font-family: "Nunito", sans-serif;
}
.mensaje-error {
  color: #f66;
  height: 1.2rem;
  margin-bottom: 1rem;
  font-size: 0.875rem;
  font-family: "Nunito", sans-serif;
}
#btn-ingresar, .boton-verde {
  width: 100%;
  padding: 0.75rem;
  background: linear-gradient(90deg, #76a73f, #9076ec);
  border: none;
  border-radius: 4px;
  color: #fff;
  font-size: 1rem;
  cursor: pointer;
  transition: all 0.3s;
  font-family: "Nunito", sans-serif;
  font-weight: 500;
  text-decoration: none;
  display: inline-block;
  text-align: center;
  box-shadow: 0 2px 5px rgba(0,0,0,0.2);
}

#btn-ingresar:hover, .boton-verde:hover {
  opacity: 0.9;
  transform: translateY(-1px);
  box-shadow: 0 4px 8px rgba(0,0,0,0.3);
}

#btn-ingresar:active, .boton-verde:active {
  transform: translateY(0);
  box-shadow: 0 1px 3px rgba(0,0,0,0.2);
}
.opciones-adicionales {
  display: flex;
  justify-content: space-between;
  margin-top: 1rem;
}
.opciones-adicionales a {
  color: #76a73f;
  font-size: 0.875rem;
  font-family: "Nunito", sans-serif;
  text-decoration: none;
}
.opciones-adicionales a:hover {
  text-decoration: underline;
}

/* Estilos para el contenedor de contraseña con ojito */
.password-input-container {
  position: relative;
  display: flex;
  align-items: center;
}

.password-input-container input {
  width: 100%;
  padding-right: 45px; /* Espacio para el botón del ojo */
}

.toggle-password {
  position: absolute;
  right: 10px;
  background: none;
  border: none;
  color: #888;
  cursor: pointer;
  font-size: 1.1rem;
  padding: 5px;
  border-radius: 3px;
  transition: color 0.3s ease;
  z-index: 1;
}

.toggle-password:hover {
  color: #76a73f;
}

.toggle-password:focus {
  outline: 2px solid #76a73f;
  outline-offset: 2px;
}

.toggle-password:active {
  transform: scale(0.95);
}

/* Icono del ojo */
.toggle-password i {
  pointer-events: none;
}

@media (min-width: 2560px) {
  .login-container {
    max-width: 600px;      /* antes era 400px */
    padding: 3rem;         /* un poco más de espacio interior */
  }

  /* Opcional: escalar texto según viewport en 4K */
  .login-container h2 {
    font-size: clamp(2.5rem, 3vw, 4rem);
  }
  .form-group label,
  .form-group input,
  #btn-ingresar {
    font-size: clamp(1rem, 1.5vw, 1.25rem);
  }
}`;

            // Enviar el contenido al servidor para guardarlo
            fetch('guardar_login_css.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/x-www-form-urlencoded',
                },
                body: 'cssContent=' + encodeURIComponent(cssContent)
            })
            .then(response => response.text())
            .then(result => {
                if (result.includes('success')) {
                    alert('Archivo login.css generado y guardado exitosamente');
                    // Recargar la página para actualizar el estado
                    location.reload();
                } else {
                    alert('Error al generar el archivo: ' + result);
                }
            })
            .catch(error => {
                console.error('Error:', error);
                alert('Error al generar el archivo: ' + error.message);
            });
        }
        
        // Función para generar config.php
        function generarConfigPHP() {
            window.location.href = 'crear_config.html';
        }
        
        // Función para generar conexion.php
        function generarConexionPHP() {
            // Contenido fijo del archivo conexion.php
            const phpContent = '<?php\n' +
'/**\n' +
' * Conexión segura a la base de datos\n' +
' * Utiliza configuración externa para mayor seguridad\n' +
' */\n\n' +
'// Cargar configuración desde archivo externo\n' +
'require_once \'config.php\';\n' +
'$db_config = $config[$environment];\n\n' +
'try {\n' +
'    // Crear conexión PDO usando configuración segura\n' +
'    $dsn_string = "mysql:host=" . $db_config["db_host"] . ";dbname=" . $db_config["db_name"] . ";charset=utf8mb4";\n' +
'    $dsn = new PDO($dsn_string, $db_config["db_user"], $db_config["db_pass"]);\n\n' +
'    // Configurar PDO para mostrar errores y usar UTF-8\n' +
'    $dsn->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);\n' +
'    $dsn->setAttribute(PDO::ATTR_DEFAULT_FETCH_MODE, PDO::FETCH_ASSOC);\n' +
'    $dsn->setAttribute(PDO::MYSQL_ATTR_INIT_COMMAND, "SET NAMES utf8mb4");\n\n' +
'} catch (PDOException $e) {\n' +
'    // En desarrollo mostrar error detallado, en producción log genérico\n' +
'    if ($db_config["debug"]) {\n' +
'        die("Error de conexión: " . $e->getMessage());\n' +
'    } else {\n' +
'        error_log("Error de conexión DB: " . $e->getMessage());\n' +
'        die("Error interno del servidor. Contacte al administrador.");\n' +
'    }\n' +
'}\n' +
'?>';

            // Enviar el contenido al servidor para guardarlo
            fetch('guardar_conexion_php.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/x-www-form-urlencoded',
                },
                body: 'phpContent=' + encodeURIComponent(phpContent)
            })
            .then(response => response.text())
            .then(result => {
                if (result.includes('success')) {
                    alert('Archivo conexion.php generado y guardado exitosamente');
                    // Recargar la página para actualizar el estado
                    location.reload();
                } else {
                    alert('Error al generar el archivo: ' + result);
                }
            })
            .catch(error => {
                console.error('Error:', error);
                alert('Error al generar el archivo: ' + error.message);
            });
        }
        
        // Función para generar login.php
        function generarLoginPHP() {
            window.location.href = 'crear_login_php.html';
        }
        
        // Función para generar session_start.php
        function generarSessionStartPHP() {
            // Contenido fijo del archivo session_start.php
            const phpContent = '<?php\n' +
'session_start();\n' +
'if (!isset($_SESSION[\'usuario\'])) {\n' +
'    header("Location: ./index.html");\n' +
'    exit();\n' +
'}\n' +
'?>';

            // Enviar el contenido al servidor para guardarlo
            fetch('guardar_session_start_php.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/x-www-form-urlencoded',
                    },
                body: 'phpContent=' + encodeURIComponent(phpContent)
            })
            .then(response => response.text())
            .then(result => {
                if (result.includes('success')) {
                    alert('Archivo session_start.php generado y guardado exitosamente');
                    // Recargar la página para actualizar el estado
                    location.reload();
                } else {
                    alert('Error al generar el archivo: ' + result);
                }
            })
            .catch(error => {
                console.error('Error:', error);
                alert('Error al generar el archivo: ' + error.message);
            });
        }
        
        // Función para generar menu.css
        function generarMenuCSS() {
            // Contenido del archivo menu.css
            const cssContent = `/* menu.css */
@import url("https://fonts.googleapis.com/css2?family=Nunito:wght@300;400;500;600;700;800&family=Tilt+Neon&display=swap");

html, body {
  height: 100%;
  margin: 0;
  padding: 0;
  font-family: "Nunito", sans-serif;
  background: radial-gradient(circle, rgba(67,69,112,1) 3%, rgba(35,36,57,1) 60%);
  scroll-behavior: smooth;
}

*, *::before, *::after {
  box-sizing: border-box;
}

/* ---------------------------------- */
/* ------------BIENVENIDA------------ */
/* ---------------------------------- */

.welcome-section {
  width: 100%;
  max-width: 1200px;
  margin: 1rem auto 0.2rem; /* top 1rem, left/right auto, bottom 0.5rem */
  padding: 0.5rem 1rem 0.2rem;
  background: rgba(255,255,255,0.1);
  border-radius: 12px;
  text-align: center;
}
.welcome-section h1 {
  font-size: clamp(1.5rem, 2vw, 2.5rem);
  color: #fff;
  margin-bottom: 0.2rem;
  font-family: "Tilt Neon", sans-serif;
 margin-top: 0rem;           /* quita espacio extra encima del h1 */
   margin-bottom: 0.5rem;   /* ajusta separación con el párrafo */

}
  
.welcome-section #user-name {
  color: #29b6f6;
}
.welcome-section .welcome-text {
  font-size: clamp(0.5rem, 1vw, 1rem);
  color: #ddd;
  line-height: 1.4;
}



/* --------------------------------------------- */
/* ------------ CONTENEDOR TARJETAS ------------ */
/* --------------------------------------------- */

.contenedor-tarjetas {
  display: flex;
  justify-content: center;
  align-items: center;
  padding: 1rem;
}


/* --------------------------------------------- */
/* -------------- MATRIZ TARJETAS -------------- */
/* --------------------------------------------- */

/* Grid de tarjetas */
.matriz-card-menu {
  display: grid;
  grid-template-columns: repeat(4, 1fr);
  gap: 2rem;
  width: 100%;
  max-width: 1200px;
}




/* --------------------------------------------- */
/* --------------- CAJA TARJETAS --------------- */
/* --------------------------------------------- */

.caja-card-menu {
  display: flex;
  flex-direction: column;
  background: rgba(255,255,255,0.1);
  border-radius: 12px;
  overflow: hidden;
  box-shadow: 0 8px 16px rgba(0,0,0,0.5);
  transition: transform 0.3s ease;
  cursor: pointer;
  border:0 solid white;
}
.caja-card-menu:hover { transform: translateY(-8px); }

/* --------------------------------------------- */
/* --------------- BADGE CURSO --------------- */
/* --------------------------------------------- */

.curso-badge {
  background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
  color: white;
  font-size: 0.75rem;
  font-weight: bold;
  padding: 4px 12px;
  border-radius: 15px;
  display: inline-block;
  margin-bottom: 8px;
  text-transform: uppercase;
  letter-spacing: 0.5px;
  box-shadow: 0 2px 4px rgba(0,0,0,0.1);
}

        
/* -------------------------------------------------- */
/* --------------- CONTENIDO TARJETAS --------------- */
/* -------------------------------------------------- */


/* Contenido de la tarjeta en 4 filas */
.contenido-card-menu {
  flex: 1;
  display: grid;
  grid-template-rows: auto auto 1fr auto;
  row-gap: 0.1rem;
  align-items: start;
  justify-items: center;
  padding: 1rem;
  text-align: center;
}
.contenido-card-menu img {
  width: 100%;
  height: 160px;
  object-fit: cover;
  border-radius: 8px;
}
.contenido-card-menu h1 {
  font-size: clamp(1.25rem, 2vw, 1.75rem);
  color: #fff;
  font-family: "Tilt Neon", sans-serif;
}
.contenido-card-menu p {
  font-size: clamp(1.1rem, 2vw, 1.3rem);
  color: #ddd;
  line-height: 1.4;
  margin-top: 0rem;
  align-self: start;
  text-align: center;
}
.contenido-card-menu p strong {
  font-size: clamp(0.8rem, 1.5vw, 1rem);
  color: #bbb;
}




/* -------------------------------------------------- */
/* ------------------ RESPONSIVIDAD ----------------- */
/* -------------------------------------------------- */




/* Ajustes extra para pantallas 4K (≥2560px) */
@media (min-width: 2560px) {
  .matriz-card-menu {
    max-width: 2000px;
    gap: 3rem;
  }
  .caja-card-menu {
    border-radius: 16px;
  }
  .contenido-card-menu {
    padding: 2.5rem;
  }
  .contenido-card-menu img {
    height: 240px;
  }
    /* Aumentar tamaño y ancho de la caja de bienvenida en 4K */
  .welcome-section {
    max-width: 2000px;
  }
  .contenido-card-menu h1 {
    font-size: clamp(2rem, 3vw, 3rem);
  }
    .welcome-section h1 {
    font-size: clamp(2.5rem, 3.5vw, 3.5rem);
  }
    .welcome-section .welcome-text {
    font-size: clamp(1.5rem, 3vw, 2rem);
  }
  .contenido-card-menu p {
    font-size: clamp(1.3rem, 2vw, 1.7rem);
  }
  .contenido-card-menu p strong {
    font-size: clamp(1rem, 1.5vw, 1.3rem);
  }

    .welcome-section {
    /* mantiene el bottom reducido */
    padding: 1rem 6rem 0.5rem;
    margin-bottom: 0.5rem;
  }
}




/* Ajustes para pantallas grandes (desde 1920px) */
@media (min-width: 1920px) {
  .welcome-section {
    margin: 1rem auto 0.5rem;
    padding: 1rem 4rem 0.5rem; /* quizá subir el padding lateral */
    max-width: 2000px;
  }
}



@media (max-width: 1024px) {
  .matriz-card-menu { grid-template-columns: repeat(2, 1fr); }
    .welcome-section {
    margin: 0.5rem auto;  /* más compacto en móvil */
    padding: 0.75rem 1rem 0.5rem;
  }
}



/* Responsive header para pantallas ≤760px */
@media (max-width: 760px) {
  
    .welcome-section {
    margin: 0.5rem auto;  /* más compacto en móvil */
    padding: 0.75rem 1rem 0.5rem;
  }
}


@media (max-width: 600px) {
  .matriz-card-menu { grid-template-columns: 1fr; }
    .welcome-section {
    margin: 0.5rem auto;  /* más compacto en móvil */
    padding: 0.75rem 1rem 0.5rem;
  }
}



@media (max-width: 400px) {
  .matriz-card-menu { padding: 0 1rem; }
    .welcome-section {
    margin: 0.5rem auto;  /* más compacto en móvil */
    padding: 0.75rem 1rem 0.5rem;
  }
}`;

            // Enviar el contenido al servidor para guardarlo
            fetch('guardar_menu_css.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/x-www-form-urlencoded',
                },
                body: 'cssContent=' + encodeURIComponent(cssContent)
            })
            .then(response => response.text())
            .then(result => {
                if (result.includes('success')) {
                    alert('Archivo menu.css generado y guardado exitosamente');
                    // Recargar la página para actualizar el estado
                    location.reload();
                } else {
                    alert('Error al generar el archivo: ' + result);
                }
            })
            .catch(error => {
                console.error('Error:', error);
                alert('Error al generar el archivo: ' + error.message);
            });
        }
        
        // Función para generar menu.php
        function generarMenuPHP() {
            // Leer el título desde index.html dinámicamente
            fetch('../index.html')
            .then(response => response.text())
            .then(html => {
                // Extraer el título del index.html
                const parser = new DOMParser();
                const doc = parser.parseFromString(html, 'text/html');
                const titleElement = doc.querySelector('.gradient-text');
                const titulo = titleElement ? titleElement.textContent.trim() : 'INTRODUCCIÓN A LOS NÚMEROS COMPLEJOS';
                
                // Enviar el título al servidor para generar menu.php
                return fetch('guardar_menu_php.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/x-www-form-urlencoded',
                    },
                    body: 'titulo=' + encodeURIComponent(titulo)
                });
            })
            .then(response => response.text())
            .then(result => {
                if (result.includes('success')) {
                    alert('Archivo menu.php generado y guardado exitosamente');
                    location.reload();
                } else {
                    alert('Error al generar el archivo: ' + result);
                }
            })
            .catch(error => {
                console.error('Error:', error);
                alert('Error al generar el archivo: ' + error.message);
            });
        }        
        // Función para generar header.css
        function generarHeaderCSS() {
            // Contenido del archivo header.css
            const cssContent = `/* Header principal */
.main-header {
  display: flex;
  flex-wrap: wrap;
  align-items: center;
  justify-content: space-between;
  padding: 0.75rem 1rem;
  background: rgba(35,36,57,0.8);
  backdrop-filter: blur(10px);
}

.main-header .logo img {
  height: 50px;
  width: auto;
}

.main-header .header-title {
  color: #fff;
  font-size: clamp(1.25rem, 5vw, 2rem);
  font-weight: 700;
  font-family: "Tilt Neon", sans-serif;
}

.main-header .session-controls .logout-button {
  color: #fff;
  background: linear-gradient(90deg, #e52d27 0%, #b31217 100%);
  padding: 0.5rem 1rem;
  border-radius: 8px;
  text-decoration: none;
  font-weight: 500;
  transition: opacity 0.3s ease;
}

.main-header .session-controls .logout-button:hover {
  opacity: 0.8;
}

/* Botón siguiente */
.boton-rojo {
  display: inline-flex;
  align-items: center;
  justify-content: center;
  color: #fff;
  background: linear-gradient(90deg, #e52d27 0%, #b31217 100%);
  padding: 0.5rem 1rem;
  border-radius: 8px;
  text-decoration: none;
  font-weight: 500;
  transition: background-position 0.3s ease, transform 0.3s ease;
  margin-top: 0rem;
  font-size: 1rem;
  position: relative;
  border: none;
  outline: none;
  box-sizing: border-box;
  line-height: 1;
  height: 2rem;
  -webkit-appearance: none;
  -moz-appearance: none;
  appearance: none;
}

.boton-rojo:hover {
  transform: scale(1.05);
}

.gradient-text {
  background: linear-gradient(
    90deg,
    #76a73f 0%,
    #338f76 40%,
    #378da7 50%,
    #7598f2 70%,
    #9076ec 100%
  );
  -webkit-background-clip: text;
  -webkit-text-fill-color: transparent;
}`;

            // Enviar el contenido al servidor para guardar
            fetch('guardar_header_css.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/x-www-form-urlencoded',
                },
                body: 'cssContent=' + encodeURIComponent(cssContent)
            })
            .then(response => response.text())
            .then(result => {
                if (result.includes('success')) {
                    alert('Archivo header.css generado y guardado exitosamente');
                    // Recargar la página para actualizar el estado
                    location.reload();
                } else {
                    alert('Error al generar el archivo: ' + result);
                }
            })
            .catch(error => {
                console.error('Error:', error);
                alert('Error al generar el archivo: ' + error.message);
            });
        }
        
        // Función para generar lectura_inscripcion_cursos.php
        function generarLecturaCursosPHP() {
            window.location.href = 'crear_lectura_inscripcion_cursos_php.html';
        }
        
        // Función para generar tabla de inscripción de cursos
        function generarTablaInscripcionCursosSQL() {
            window.location.href = 'crear_tabla_inscripcion_cursos_sql.html';
        }
        
        // Función para generar editar tabla de inscripción de cursos
        function generarEditarTablaInscripcionCursosSQL() {
            window.location.href = 'crear_editar_tabla_inscripcion_cursos_sql.html';
        }
        
        function generarCrearLeerRolPHP() {
            window.location.href = 'crear_leer_rol.html';
        }
        
        function generarConfiguracionCursosPHP() {
            window.location.href = 'crear_configuracion_cursos_php.html';
        }
        
        // Función para generar logout.php
        // Función para generar logout.php
        function generarLogoutPHP() {
            fetch('guardar_logout_php.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/x-www-form-urlencoded',
                },
                body: 'phpContent='
            })
            .then(response => response.text())
            .then(result => {
                if (result.includes('success')) {
                    alert('Archivo logout.php generado y guardado exitosamente');
                    location.reload();
                } else {
                    alert('Error al generar el archivo: ' + result);
                }
            })
            .catch(error => {
                console.error('Error:', error);
                alert('Error al generar el archivo: ' + error.message);
            });
        }
        // Función para manejar el estado manual de los archivos
        function toggleManualStatus(filename, isChecked) {
            const storageKey = 'manual_' + filename.replace(/\./g, '_');
            
            if (isChecked) {
                localStorage.setItem(storageKey, 'exists');
                console.log(`Archivo "${filename}" marcado como EXISTE.`);
            } else {
                localStorage.setItem(storageKey, 'missing');
                console.log(`Archivo "${filename}" marcado como NO EXISTE.`);
            }
            
            // Recargar la página para aplicar el cambio
            location.reload();
        }
        
    </script>
</body>
</html>
