<?php
session_start();

// Incluir configuración de cursos
require_once 'configuracion_cursos.php';

// Función para verificar si existe un archivo de actividad
function verificarArchivoActividad($curso, $unidad, $actividad) {
    // Extraer el número de la unidad (ej: "Unidad 1" -> "01", "Unidad 2" -> "02")
    preg_match('/Unidad (\d+)/', $unidad, $matches);
    $numero_unidad = isset($matches[1]) ? sprintf('%02d', $matches[1]) : '01';
    $archivo = "recursos/{$curso}_u{$numero_unidad}_act" . sprintf('%02d', $actividad) . ".php";
    return file_exists($archivo);
}

// Función para obtener el estado de un archivo
function obtenerEstadoArchivo($curso, $unidad, $actividad) {
    if (verificarArchivoActividad($curso, $unidad, $actividad)) {
        return 'existe';
    } else {
        return 'no_existe';
    }
}
?>

<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Panel de Control - Gestión de Cursos</title>
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css" rel="stylesheet"/>
    <link rel="stylesheet" href="general.css" />
    <style>
        body {
            font-family: "Nunito", sans-serif;
            background: radial-gradient(circle, rgba(67,69,112,1) 3%, rgba(35,36,57,1) 60%);
            min-height: 100vh;
            margin: 0;
            padding: 0;
        }

        .panel-container {
            max-width: 1400px;
            margin: 0 auto;
            padding: 2rem;
        }

        .panel-header {
            text-align: center;
            margin-bottom: 2rem;
        }

        .panel-title {
            color: #fff;
            font-size: 2.5rem;
            font-family: "Tilt Neon", sans-serif;
            margin-bottom: 0.5rem;
        }

        .panel-subtitle {
            color: #ddd;
            font-size: 1.1rem;
            margin-bottom: 2rem;
        }

        .back-button {
            position: fixed;
            top: 2rem;
            left: 2rem;
            z-index: 1000;
        }

        .curso-container {
            background: rgba(255,255,255,0.1);
            border-radius: 12px;
            margin-bottom: 2rem;
            overflow: hidden;
            border: 2px solid transparent;
            transition: all 0.3s ease;
        }

        .curso-container:hover {
            border-color: #29b6f6;
            box-shadow: 0 10px 25px rgba(41, 182, 246, 0.3);
        }

        .curso-header {
            padding: 1.5rem;
            background: linear-gradient(135deg, rgba(102, 126, 234, 0.2) 0%, rgba(118, 75, 162, 0.2) 100%);
            border-bottom: 1px solid rgba(255,255,255,0.1);
            cursor: pointer;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }

        .curso-info {
            flex: 1;
        }

        .curso-titulo {
            color: #fff;
            font-size: 1.4rem;
            font-weight: 600;
            margin-bottom: 0.5rem;
            display: flex;
            align-items: center;
            gap: 0.5rem;
        }

        .curso-codigo {
            color: #4fc3f7;
            font-weight: 700;
        }

        .curso-descripcion {
            color: #ddd;
            font-size: 0.9rem;
            line-height: 1.4;
            margin-bottom: 0.5rem;
        }

        .curso-meta {
            display: flex;
            gap: 1rem;
            align-items: center;
            flex-wrap: wrap;
        }

        .curso-estado {
            padding: 0.25rem 0.75rem;
            border-radius: 15px;
            font-size: 0.8rem;
            font-weight: 600;
            text-transform: uppercase;
        }

        .estado-disponible { background: #4caf50; color: #fff; }
        .estado-mantenimiento { background: #ff9800; color: #fff; }
        .estado-no-disponible { background: #f44336; color: #fff; }
        .estado-proximamente { background: #9c27b0; color: #fff; }

        .curso-fecha {
            color: #bbb;
            font-size: 0.85rem;
        }

        .curso-toggle {
            color: #4fc3f7;
            font-size: 1.5rem;
            transition: transform 0.3s ease;
        }

        .curso-toggle.expanded {
            transform: rotate(180deg);
        }

        .curso-content {
            display: none;
            padding: 1.5rem;
        }

        .curso-content.expanded {
            display: block;
        }

        .unidad-container {
            background: rgba(255,255,255,0.05);
            border-radius: 8px;
            margin-bottom: 1rem;
            border-left: 4px solid #f093fb;
        }

        .unidad-header {
            padding: 1rem;
            cursor: pointer;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }

        .unidad-info {
            flex: 1;
        }

        .unidad-titulo {
            color: #fff;
            font-size: 1.1rem;
            font-weight: 600;
            margin-bottom: 0.25rem;
            display: flex;
            align-items: center;
            gap: 0.5rem;
        }

        .unidad-descripcion {
            color: #ddd;
            font-size: 0.85rem;
            margin-bottom: 0.5rem;
        }

        .unidad-meta {
            display: flex;
            gap: 1rem;
            align-items: center;
            flex-wrap: wrap;
        }

        .unidad-duracion {
            color: #4fc3f7;
            font-size: 0.8rem;
            font-weight: 600;
        }

        .unidad-actividades {
            color: #bbb;
            font-size: 0.8rem;
        }

        .unidad-toggle {
            color: #f093fb;
            font-size: 1.2rem;
            transition: transform 0.3s ease;
        }

        .unidad-toggle.expanded {
            transform: rotate(180deg);
        }

        .unidad-content {
            display: none;
            padding: 1rem;
            border-top: 1px solid rgba(255,255,255,0.1);
        }

        .unidad-content.expanded {
            display: block;
        }

        .aprendizaje-esperado {
            background: rgba(79, 195, 247, 0.1);
            border-radius: 6px;
            padding: 0.75rem;
            margin-bottom: 1rem;
            border-left: 3px solid #4fc3f7;
        }

        .aprendizaje-titulo {
            color: #4fc3f7;
            font-size: 0.85rem;
            font-weight: 600;
            margin-bottom: 0.25rem;
        }

        .aprendizaje-texto {
            color: #ddd;
            font-size: 0.8rem;
            line-height: 1.4;
        }

        .actividades-grid {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(300px, 1fr));
            gap: 1rem;
        }

        .actividad-card {
            background: rgba(255,255,255,0.05);
            border-radius: 8px;
            padding: 1rem;
            border: 2px solid transparent;
            transition: all 0.3s ease;
        }

        .actividad-card:hover {
            border-color: #4facfe;
            transform: translateY(-2px);
        }

        .actividad-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 0.75rem;
        }

        .actividad-titulo {
            color: #fff;
            font-size: 1rem;
            font-weight: 600;
            display: flex;
            align-items: center;
            gap: 0.5rem;
        }

        .actividad-archivo {
            color: #4facfe;
            font-size: 0.8rem;
            font-family: monospace;
            background: rgba(79, 172, 254, 0.1);
            padding: 0.25rem 0.5rem;
            border-radius: 4px;
        }

        .actividad-estado {
            padding: 0.2rem 0.6rem;
            border-radius: 12px;
            font-size: 0.7rem;
            font-weight: 600;
            text-transform: uppercase;
        }

        .estado-existe { background: #4caf50; color: #fff; }
        .estado-no-existe { background: #f44336; color: #fff; }

        .actividad-acciones {
            display: flex;
            gap: 0.5rem;
            flex-wrap: wrap;
        }

        .btn-accion {
            padding: 0.4rem 0.8rem;
            border: none;
            border-radius: 6px;
            font-size: 0.75rem;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            display: flex;
            align-items: center;
            gap: 0.25rem;
        }

        .btn-editar {
            background: #4caf50;
            color: #fff;
        }

        .btn-editar:hover {
            background: #45a049;
            transform: translateY(-1px);
        }

        .btn-preview {
            background: #2196f3;
            color: #fff;
        }

        .btn-preview:hover {
            background: #1976d2;
            transform: translateY(-1px);
        }

        .btn-codigo {
            background: #ff9800;
            color: #fff;
        }

        .btn-codigo:hover {
            background: #f57c00;
            transform: translateY(-1px);
        }

        .btn-database {
            background: #607d8b;
            color: #fff;
        }

        .btn-database:hover {
            background: #455a64;
            transform: translateY(-1px);
        }

        .btn-crear {
            background: #9c27b0;
            color: #fff;
        }

        .btn-crear:hover {
            background: #7b1fa2;
            transform: translateY(-1px);
        }

        .btn-disabled {
            background: #666;
            color: #999;
            cursor: not-allowed;
        }

        /* Modal de previsualización */
        .modal-overlay {
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(0, 0, 0, 0.8);
            z-index: 10000;
            display: none;
            align-items: center;
            justify-content: center;
            padding: 2rem;
        }

        .modal-overlay.active {
            display: flex;
        }

        .modal-content {
            background: white;
            border-radius: 12px;
            width: 100%;
            max-width: 1200px;
            max-height: 90vh;
            overflow: hidden;
            box-shadow: 0 20px 40px rgba(0, 0, 0, 0.3);
            position: relative;
        }

        .modal-header {
            background: #1e40af;
            color: white;
            padding: 1rem 1.5rem;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }

        .modal-title {
            font-size: 1.2rem;
            font-weight: 600;
            margin: 0;
        }

        .modal-close {
            background: none;
            border: none;
            color: white;
            font-size: 1.5rem;
            cursor: pointer;
            padding: 0.5rem;
            border-radius: 50%;
            transition: background 0.3s ease;
        }

        .modal-close:hover {
            background: rgba(255, 255, 255, 0.2);
        }

        .modal-body {
            height: calc(90vh - 80px);
            overflow-y: auto;
        }

        .modal-iframe {
            width: 100%;
            height: 100%;
            border: none;
        }

        .estadisticas-curso {
            background: rgba(255,255,255,0.05);
            border-radius: 8px;
            padding: 1rem;
            margin-bottom: 1rem;
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(150px, 1fr));
            gap: 1rem;
        }

        .stat-item {
            text-align: center;
        }

        .stat-numero {
            color: #4fc3f7;
            font-size: 1.5rem;
            font-weight: 700;
        }

        .stat-label {
            color: #bbb;
            font-size: 0.8rem;
            text-transform: uppercase;
        }

        @media (max-width: 768px) {
            .panel-container {
                padding: 1rem;
            }
            
            .curso-header {
                flex-direction: column;
                align-items: flex-start;
                gap: 1rem;
            }
            
            .curso-meta {
                width: 100%;
                justify-content: space-between;
            }
            
            .actividades-grid {
                grid-template-columns: 1fr;
            }
            
            .back-button {
                position: static;
                margin-bottom: 1rem;
            }
        }
    </style>
</head>

<body>
    <!-- Botón de regreso -->
    <div class="back-button">
        <a href="#" onclick="volverAlChecklist()" class="boton-azul">
            <i class="fas fa-arrow-left"></i>
            <span>Volver al Checklist</span>
        </a>
    </div>

    <div class="panel-container">
        <!-- Header del Panel -->
        <div class="panel-header">
            <h1 class="panel-title">
                <i class="fas fa-cogs"></i>
                Panel de Control de Cursos
            </h1>
            <p class="panel-subtitle">
                Gestión completa de cursos, unidades y actividades basada en configuración automática
            </p>
        </div>

        <!-- Lista de Cursos -->
        <?php foreach ($cursos as $codigo_interno => $curso): ?>
            <div class="curso-container">
                <!-- Header del Curso -->
                <div class="curso-header" onclick="toggleCurso('<?php echo $codigo_interno; ?>')">
                    <div class="curso-info">
                        <div class="curso-titulo">
                            <i class="fas fa-graduation-cap"></i>
                            <span>📚 CURSO (<span class="curso-codigo"><?php echo $codigo_interno; ?></span>): <?php echo $curso['nombre_curso']; ?> (<?php echo $curso['codigo_curso']; ?>)</span>
                        </div>
                        <div class="curso-descripcion">
                            <?php echo $curso['descripcion']; ?>
                        </div>
                        <div class="curso-meta">
                            <span class="curso-estado estado-<?php echo strtolower(str_replace(' ', '-', $curso['estado'])); ?>">
                                <?php echo $curso['estado']; ?>
                            </span>
                            <span class="curso-fecha">
                                <i class="fas fa-calendar"></i> <?php echo $curso['fecha_inicio']; ?>
                            </span>
                        </div>
                    </div>
                    <div class="curso-toggle" id="toggle-<?php echo $codigo_interno; ?>">
                        <i class="fas fa-chevron-down"></i>
                    </div>
                </div>

                <!-- Contenido del Curso -->
                <div class="curso-content" id="content-<?php echo $codigo_interno; ?>">
                    <!-- Estadísticas del Curso -->
                    <?php 
                    $total_unidades = count($curso['unidades']);
                    $total_actividades = array_sum(array_column($curso['unidades'], 'actividades'));
                    $actividades_existentes = 0;
                    foreach ($curso['unidades'] as $unidad_nombre => $unidad) {
                        for ($i = 1; $i <= $unidad['actividades']; $i++) {
                            if (verificarArchivoActividad($codigo_interno, $unidad_nombre, $i)) {
                                $actividades_existentes++;
                            }
                        }
                    }
                    ?>
                    <div class="estadisticas-curso">
                        <div class="stat-item">
                            <div class="stat-numero"><?php echo $total_unidades; ?></div>
                            <div class="stat-label">Unidades</div>
                        </div>
                        <div class="stat-item">
                            <div class="stat-numero"><?php echo $total_actividades; ?></div>
                            <div class="stat-label">Actividades</div>
                        </div>
                        <div class="stat-item">
                            <div class="stat-numero"><?php echo $actividades_existentes; ?></div>
                            <div class="stat-label">Creadas</div>
                        </div>
                        <div class="stat-item">
                            <div class="stat-numero"><?php echo round(($actividades_existentes / $total_actividades) * 100); ?>%</div>
                            <div class="stat-label">Progreso</div>
                        </div>
                    </div>

                    <!-- Unidades del Curso -->
                    <?php foreach ($curso['unidades'] as $unidad_nombre => $unidad): ?>
                        <div class="unidad-container">
                            <!-- Header de la Unidad -->
                            <div class="unidad-header" onclick="toggleUnidad('<?php echo $codigo_interno; ?>-<?php echo $unidad_nombre; ?>')">
                                <div class="unidad-info">
                                    <div class="unidad-titulo">
                                        <i class="fas fa-book"></i>
                                        <?php echo $unidad_nombre; ?>: <?php echo $unidad['nombre']; ?>
                                    </div>
                                    <div class="unidad-descripcion">
                                        <?php echo $unidad['aprendizaje_esperado']; ?>
                                    </div>
                                    <div class="unidad-meta">
                                        <span class="unidad-duracion">
                                            <i class="fas fa-clock"></i> <?php echo $unidad['duracion']; ?>
                                        </span>
                                        <span class="unidad-actividades">
                                            <i class="fas fa-tasks"></i> <?php echo $unidad['actividades']; ?> actividades
                                        </span>
                                    </div>
                                </div>
                                <div class="unidad-toggle" id="toggle-unidad-<?php echo $codigo_interno; ?>-<?php echo $unidad_nombre; ?>">
                                    <i class="fas fa-chevron-down"></i>
                                </div>
                            </div>

                            <!-- Contenido de la Unidad -->
                            <div class="unidad-content" id="content-unidad-<?php echo $codigo_interno; ?>-<?php echo $unidad_nombre; ?>">
                                <!-- Aprendizaje Esperado -->
                                <div class="aprendizaje-esperado">
                                    <div class="aprendizaje-titulo">
                                        <i class="fas fa-target"></i> Aprendizaje Esperado
                                    </div>
                                    <div class="aprendizaje-texto">
                                        <?php echo $unidad['aprendizaje_esperado']; ?>
                                    </div>
                                </div>

                                <!-- Criterios de Evaluación -->
                                <?php if (!empty($unidad['criterios_evaluacion'])): ?>
                                    <div class="aprendizaje-esperado">
                                        <div class="aprendizaje-titulo">
                                            <i class="fas fa-check-circle"></i> Criterios de Evaluación
                                        </div>
                                        <div class="aprendizaje-texto">
                                            <?php foreach ($unidad['criterios_evaluacion'] as $codigo => $criterio): ?>
                                                <div style="margin-bottom: 0.5rem;">
                                                    <strong><?php echo $codigo; ?>:</strong> <?php echo $criterio; ?>
                                                </div>
                                            <?php endforeach; ?>
                                        </div>
                                    </div>
                                <?php endif; ?>

                                <!-- Actividades -->
                                <div class="actividades-grid">
                                    <?php for ($i = 1; $i <= $unidad['actividades']; $i++): ?>
                                        <?php 
                                        // Extraer el número de la unidad con formato correcto
                                        preg_match('/Unidad (\d+)/', $unidad_nombre, $matches);
                                        $numero_unidad = isset($matches[1]) ? sprintf('%02d', $matches[1]) : '01';
                                        $archivo_actividad = "{$codigo_interno}_u{$numero_unidad}_act" . sprintf('%02d', $i) . ".php";
                                        $estado_archivo = obtenerEstadoArchivo($codigo_interno, $unidad_nombre, $i);
                                        ?>
                                        <div class="actividad-card">
                                            <div class="actividad-header">
                                                <div class="actividad-titulo">
                                                    <i class="fas fa-tasks"></i>
                                                    Actividad <?php echo $i; ?>
                                                </div>
                                                <span class="actividad-estado estado-<?php echo $estado_archivo; ?>">
                                                    <?php echo $estado_archivo === 'existe' ? 'Existe' : 'No Existe'; ?>
                                                </span>
                                            </div>
                                            <div class="actividad-archivo">
                                                <?php echo $archivo_actividad; ?>
                                            </div>
                                            <div class="actividad-acciones">
                                                <?php if ($estado_archivo === 'existe'): ?>
                                                    <button class="btn-accion btn-codigo" onclick="crearTablaAvance('<?php echo $codigo_interno; ?>', '<?php echo $unidad_nombre; ?>', <?php echo $i; ?>)">
                                                        <i class="fas fa-database"></i> Crear tabla
                                                    </button>
                                                    <button class="btn-accion btn-editar" onclick="editarActividad('<?php echo $archivo_actividad; ?>')">
                                                        <i class="fas fa-edit"></i> Editar
                                                    </button>
                                                    <button class="btn-accion btn-preview" onclick="previsualizarActividad('<?php echo $codigo_interno; ?>', '<?php echo $unidad_nombre; ?>', <?php echo $i; ?>)">
                                                        <i class="fas fa-eye"></i> Preview
                                                    </button>
                                                    <button class="btn-accion btn-database" onclick="verBaseDatos('<?php echo $codigo_interno; ?>', '<?php echo $unidad_nombre; ?>')">
                                                        <i class="fas fa-table"></i> Ver BD
                                                    </button>
                                                <?php else: ?>
                                                    <button class="btn-accion btn-codigo" onclick="crearTablaAvance('<?php echo $codigo_interno; ?>', '<?php echo $unidad_nombre; ?>', <?php echo $i; ?>)">
                                                        <i class="fas fa-database"></i> Crear tabla
                                                    </button>
                                                    <button class="btn-accion btn-crear" onclick="crearActividad('<?php echo $archivo_actividad; ?>')">
                                                        <i class="fas fa-plus"></i> Crear
                                                    </button>
                                                    <button class="btn-accion btn-disabled" disabled>
                                                        <i class="fas fa-eye-slash"></i> Preview
                                                    </button>
                                                    <button class="btn-accion btn-database" onclick="verBaseDatos('<?php echo $codigo_interno; ?>', '<?php echo $unidad_nombre; ?>')">
                                                        <i class="fas fa-table"></i> Ver BD
                                                    </button>
                                                <?php endif; ?>
                                            </div>
                                        </div>
                                    <?php endfor; ?>
                                </div>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>
            </div>
        <?php endforeach; ?>
    </div>

    <!-- Modal de Previsualización -->
    <div class="modal-overlay" id="preview-modal">
        <div class="modal-content">
            <div class="modal-header">
                <h3 class="modal-title" id="modal-title">Previsualización de Actividad</h3>
                <button class="modal-close" onclick="cerrarModal()">
                    <i class="fas fa-times"></i>
                </button>
            </div>
            <div class="modal-body">
                <iframe class="modal-iframe" id="modal-iframe" src=""></iframe>
            </div>
        </div>
    </div>

    <script>
        // Función para volver al checklist en la misma posición
        function volverAlChecklist() {
            if (document.referrer && document.referrer.includes('ayuda.php')) {
                history.back();
            } else {
                window.location.href = 'generador_de_archivos/ayuda.php';
            }
        }

        // Función para expandir/contraer cursos
        function toggleCurso(codigo) {
            const content = document.getElementById('content-' + codigo);
            const toggle = document.getElementById('toggle-' + codigo);
            
            if (content.classList.contains('expanded')) {
                content.classList.remove('expanded');
                toggle.classList.remove('expanded');
            } else {
                content.classList.add('expanded');
                toggle.classList.add('expanded');
            }
        }

        // Función para expandir/contraer unidades
        function toggleUnidad(id) {
            const content = document.getElementById('content-unidad-' + id);
            const toggle = document.getElementById('toggle-unidad-' + id);
            
            if (content.classList.contains('expanded')) {
                content.classList.remove('expanded');
                toggle.classList.remove('expanded');
            } else {
                content.classList.add('expanded');
                toggle.classList.add('expanded');
            }
        }

        // Función para editar actividad
        function editarActividad(archivo) {
            // Guardar el estado actual de los despliegues antes de ir al editor
            const expandedElements = document.querySelectorAll('.expanded');
            const expandedIds = Array.from(expandedElements).map(el => el.id);
            localStorage.setItem('panel_expanded_elements', JSON.stringify(expandedIds));
            
            // Extraer información del archivo (ej: c001_u01_act01.php)
            const partes = archivo.replace('.php', '').split('_');
            const curso = partes[0];
            const unidad = `Unidad ${partes[1].substring(1)}`;
            const actividad = partes[2].substring(3);
            
            // Redirigir al panel de configuración
            window.location.href = `api/crear_editar_actividad.php?curso=${curso}&unidad=${encodeURIComponent(unidad)}&actividad=${actividad}&modo=editar`;
        }

        // Función para previsualizar actividad en modal
        function previsualizarActividad(curso, unidad, actividad) {
            // Extraer número de unidad del string "Unidad X"
            const unidadNumero = unidad.replace('Unidad ', '');
            const url = `curso.php?curso=${curso}&unidad=${unidadNumero}&recurso=${actividad}&preview=1`;
            const modal = document.getElementById('preview-modal');
            const iframe = document.getElementById('modal-iframe');
            const title = document.getElementById('modal-title');
            
            // Configurar el modal
            title.textContent = `Previsualización: ${curso} - ${unidad} - Actividad ${actividad}`;
            iframe.src = url;
            
            // Mostrar el modal
            modal.classList.add('active');
            
            // Prevenir scroll del body
            document.body.style.overflow = 'hidden';
        }

        // Función para cerrar el modal
        function cerrarModal() {
            const modal = document.getElementById('preview-modal');
            const iframe = document.getElementById('modal-iframe');
            
            // Ocultar el modal
            modal.classList.remove('active');
            
            // Limpiar el iframe
            iframe.src = '';
            
            // Restaurar scroll del body
            document.body.style.overflow = 'auto';
        }

        // Cerrar modal al hacer clic fuera del contenido
        document.getElementById('preview-modal').addEventListener('click', function(e) {
            if (e.target === this) {
                cerrarModal();
            }
        });

        // Cerrar modal con tecla Escape
        document.addEventListener('keydown', function(e) {
            if (e.key === 'Escape') {
                cerrarModal();
            }
        });

        // Función para ver código
        function verCodigo(archivo) {
            window.open('recursos/' + archivo, '_blank');
        }

        // Función para crear actividad
        function crearActividad(archivo) {
            // Extraer información del archivo (ej: c001_u01_act01.php)
            const partes = archivo.replace('.php', '').split('_');
            const curso = partes[0];
            const unidad = partes[1].substring(1); // Extraer número sin "u"
            const actividad = partes[2].substring(3); // Extraer número sin "act"
            
            // Confirmar acción
            if (!confirm(`¿Crear archivo de actividad para ${curso} - Unidad ${unidad} - Actividad ${actividad}?`)) {
                return;
            }

            // Mostrar loading
            const btn = event.target;
            const originalText = btn.innerHTML;
            btn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Creando...';
            btn.disabled = true;

            // Llamar al endpoint para crear el archivo
            fetch('api/crear_archivo_actividad.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify({
                    curso: curso,
                    unidad: unidad.padStart(2, '0'), // Formato con 2 dígitos
                    actividad: actividad
                })
            })
            .then(response => {
                console.log('Response status:', response.status);
                console.log('Response headers:', response.headers);
                
                if (!response.ok) {
                    throw new Error(`HTTP ${response.status}: ${response.statusText}`);
                }
                
                return response.json();
            })
            .then(data => {
                console.log('Response data:', data);
                
                if (data.success) {
                    alert(`✅ ${data.message}`);
                    
                    // Actualizar el estado del botón sin recargar la página
                    const btnContainer = btn.parentElement;
                    const archivo = btn.getAttribute('onclick').match(/'([^']+)'/)[1];
                    
                    // Obtener referencias antes de cambiar el HTML
                    const actividadCard = btn.closest('.actividad-card');
                    const estadoElement = actividadCard ? actividadCard.querySelector('.actividad-estado') : null;
                    
                    // Cambiar el botón "Crear" por "Editar"
                    btnContainer.innerHTML = `
                        <button class="btn-accion btn-codigo" onclick="crearTablaAvance('${curso}', 'Unidad ${unidad}', ${actividad})">
                            <i class="fas fa-database"></i> Crear tabla
                        </button>
                        <button class="btn-accion btn-editar" onclick="editarActividad('${archivo}')">
                            <i class="fas fa-edit"></i> Editar
                        </button>
                        <button class="btn-accion btn-preview" onclick="previsualizarActividad('${curso}', 'Unidad ${unidad}', ${actividad})">
                            <i class="fas fa-eye"></i> Preview
                        </button>
                        <button class="btn-accion btn-database" onclick="verBaseDatos('${curso}', 'Unidad ${unidad}')">
                            <i class="fas fa-table"></i> Ver BD
                        </button>
                    `;
                    
                    // Actualizar el estado visual de la actividad si existe
                    if (estadoElement) {
                        estadoElement.textContent = 'Existe';
                        estadoElement.className = 'actividad-estado estado-existe';
                    }
                    
                } else {
                    alert(`❌ Error: ${data.message}`);
                }
            })
            .catch(error => {
                console.error('Error completo:', error);
                alert(`❌ Error de conexión: ${error.message}`);
            })
            .finally(() => {
                // Restaurar botón
                btn.innerHTML = originalText;
                btn.disabled = false;
            });
        }

        // Función para ver variables de base de datos
        function verBaseDatos(curso, unidad) {
            // Extraer número de unidad
            const unidadNumero = unidad.replace('Unidad ', '').padStart(2, '0');
            
            // Redirigir al visor de base de datos
            window.location.href = `api/ver_base_datos.php?curso=${curso}&unidad=${unidadNumero}`;
        }

        // Función para crear tabla de avance
        function crearTablaAvance(curso, unidad, actividad) {
            // Confirmar acción
            if (!confirm(`¿Crear/actualizar tabla de avance para ${curso} - ${unidad} - Actividad ${actividad}?`)) {
                return;
            }

            // Mostrar loading
            const btn = event.target;
            const originalText = btn.innerHTML;
            btn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Procesando...';
            btn.disabled = true;

            // Extraer número de unidad
            const unidadNumero = unidad.replace('Unidad ', '').padStart(2, '0');
            
            // Llamar al endpoint
            fetch('api/crear_tabla_avance.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify({
                    curso: curso,
                    unidad: unidadNumero,
                    actividad: actividad
                })
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    alert(`✅ ${data.message}`);
                } else {
                    alert(`❌ Error: ${data.message}`);
                }
            })
            .catch(error => {
                console.error('Error:', error);
                alert('❌ Error de conexión');
            })
            .finally(() => {
                // Restaurar botón
                btn.innerHTML = originalText;
                btn.disabled = false;
            });
        }

        // Función para restaurar el estado de los despliegues
        function restaurarEstadoDespliegues() {
            const expandedIds = localStorage.getItem('panel_expanded_elements');
            if (expandedIds) {
                try {
                    const ids = JSON.parse(expandedIds);
                    ids.forEach(id => {
                        const element = document.getElementById(id);
                        if (element) {
                            element.classList.add('expanded');
                        }
                    });
                    // Limpiar el localStorage después de restaurar
                    localStorage.removeItem('panel_expanded_elements');
                } catch (e) {
                    console.error('Error al restaurar estado:', e);
                }
            }
        }

        // Función para expandir todo por defecto
        function expandirTodoPorDefecto() {
            // Expandir todos los cursos
            document.querySelectorAll('.curso-content').forEach(content => {
                content.classList.add('expanded');
            });
            
            // Expandir todos los toggles de cursos
            document.querySelectorAll('.curso-toggle').forEach(toggle => {
                toggle.classList.add('expanded');
            });
            
            // Expandir todas las unidades
            document.querySelectorAll('.unidad-content').forEach(content => {
                content.classList.add('expanded');
            });
            
            // Expandir todos los toggles de unidades
            document.querySelectorAll('.unidad-toggle').forEach(toggle => {
                toggle.classList.add('expanded');
            });
        }

        // Restaurar estado al cargar la página
        document.addEventListener('DOMContentLoaded', function() {
            // Primero intentar restaurar estado guardado
            restaurarEstadoDespliegues();
            
            // Si no hay estado guardado, expandir todo por defecto
            const expandedIds = localStorage.getItem('panel_expanded_elements');
            if (!expandedIds) {
                expandirTodoPorDefecto();
            }
        });

    </script>
</body>
</html>